## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ----installation-dev, eval=FALSE---------------------------------------------
# # Install devtools from CRAN
# if (!require("devtools", quietly = TRUE)) {
#     install.packages("devtools")
# }
# 
# # Install the development version of OAtools from the UW Virology NGS GitHub
# devtools::install_github(
#     repo = "uwvirology-ngs/OAtools",
#     dependencies = TRUE,
#     build_vignettes = TRUE
# )

## ----installation-bioconductor, eval=FALSE------------------------------------
# # install Bioconductor from CRAN
# if (!require("BiocManager", quietly = TRUE)) {
#     install.packages("BiocManager")
# }
# 
# # install OAtools from Bioconductor
# BiocManager::install("OAtools")

## ----libraries, message=FALSE-------------------------------------------------
# attach OAtools and SummarizedExperiment libraries
library(OAtools)
library(SummarizedExperiment)

## ----install-dplyr, eval=FALSE------------------------------------------------
# # install dplyr from CRAN
# if (!require("dplyr", quietly = TRUE)) {
#     install.packages("dplyr")
# }

## ----data_import--------------------------------------------------------------
# save filepath to example OpenArray gene expression run data
path = system.file(
    "extdata", 
    "oa_gene_expression_1.xlsx", 
    package = "OAtools"
)

# transform the run data into a SummarizedExperiment
se <- excelToSE(excel_path = path)

## ----structure-assay_matrix---------------------------------------------------
# retrieve the assay matrix and display a subset as a data frame
as.data.frame(assays(se)$fluo_reporter) |> 
    dplyr::select(well_2321:well_2386) |>
    dplyr::slice(11:20)

## ----structure-coldata--------------------------------------------------------
# retrieve the coldata and render a subset as a data frame
as.data.frame(colData(se)) |> 
    dplyr::select(
        well, sample_name:target_name, reporter, crt, amp_score:amp_status
    ) |> 
    dplyr::slice(1:10)

## ----structure-rowdata--------------------------------------------------------
# retrieve the rowdata and render a subset as a data frame
as.data.frame(rowData(se)) |> 
    dplyr::slice(1:10)

## ----structure-metadata-------------------------------------------------------
# render the experiment level metadata
metadata(se)$run_info

## ----install-kableExtra, eval=FALSE-------------------------------------------
# # install kableExtra from CRAN
# if (!require("kableExtra", quietly = TRUE)) {
#     install.packages("kableExtra")
# }

## ----reporting-results, eval=FALSE--------------------------------------------
# # generate an HTML report from the run data
# generateReport(se = se)

## ----regression-data_import---------------------------------------------------
# clear the environment
rm(list = ls())

# save filepath to example OpenArray gene expression run data
path = system.file(
    "extdata", 
    "oa_gene_expression_1.xlsx", 
    package = "OAtools"
)

# transform the run data into a SummarizedExperiment
se <- excelToSE(excel_path = path)

## ----regression-fit_curve-----------------------------------------------------
# optimize model curves to each PCR reaction
se <- computeModels(
    se = se,
    assay_name = "fluo_reporter"
)

# display an example model, fit to the PCR reaction in well 2345
metadata(se)$fluo_reporter_models$well_2345

## ----regression-derive_results------------------------------------------------
# save filepath to target-threshold-key
key_path = system.file(
    "extdata", 
    "target_threshold_key.xlsx", 
    package = "OAtools"
)

# assign a PCR result according to the key
se <- determinePCRResults(
    se = se, 
    key_path = key_path
)

# render a snapshot including the logic for determining the result
as.data.frame(SummarizedExperiment::colData(se)) |> 
    dplyr::select(
        well, crt, midpoint_slope, delta_fluo, 
        crt_threshold, slope_threshold, delta_threshold, result
    ) |> 
    dplyr::slice_head(n = 10)

## ----interop-NormqPCR_setup, message=FALSE, eval=FALSE------------------------
# # Install ReadqPCR from Bioconductor
# if (!require("ReadqPCR", quietly = TRUE)) {
#     BiocManager::install("ReadqPCR")
# }
# 
# # Install NormqPCR from Bioconductor
# if (!require("NormqPCR", quietly = TRUE)) {
#     BiocManager::install("NormqPCR")
# }

## ----interop-NormqPCR---------------------------------------------------------
# convert SummarizedExperiment container to qPCRBatch
qpcr <- seToQPCRBatch(se)

# choose housekeeping gene (in this case, the RNAse P control)
housekeeping_gene = "RNAse_P_Pa04930436_g1"

# run delta-Cq calculation
norm <- NormqPCR::deltaCq(
    qPCRBatch = qpcr,
    hkgs = housekeeping_gene
)

# display expression matrix
as.data.frame(Biobase::exprs(norm)) |> 
    dplyr::select(`Pos_Control_A`:`Sample-106`) |> 
    knitr::kable(digits = 2)
    

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

