## ----style, echo = FALSE, results = 'asis'------------------------------------
  BiocStyle::markdown()

## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE, eval = TRUE, message = FALSE, 
    warning=FALSE, error = TRUE)

## ----eval=FALSE---------------------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE))
#     install.packages("BiocManager")
# BiocManager::install("granulator")

## ----eval=TRUE----------------------------------------------------------------
library(granulator)

## ----results='hide'-----------------------------------------------------------
# load datasets for deconvolution of PBMC RNA-seq data
load_ABIS()

## -----------------------------------------------------------------------------
# print TPM values in bulk RNA-seq
bulkRNAseq_ABIS[1:5, 1:5]

## -----------------------------------------------------------------------------
# print TPM values in reference profile matrix
sigMatrix_ABIS_S0[1:5, 1:5]

## -----------------------------------------------------------------------------
# print measured cell type proportions (percentages)
groundTruth_ABIS[1:5, 1:5]

## -----------------------------------------------------------------------------
# create list if multiple signature matrices to test simultaneously
sigList = list(
  ABIS_S0 = sigMatrix_ABIS_S0,
  ABIS_S1 = sigMatrix_ABIS_S1, 
  ABIS_S2 = sigMatrix_ABIS_S2, 
  ABIS_S3 = sigMatrix_ABIS_S3)

## ----fig.retina = 1-----------------------------------------------------------
# plot signature matrix similarity matrices
plot_similarity(sigMatrix=sigList)

## ----results='hide'-----------------------------------------------------------
# deconvolute input data using all available methods by default
decon <- deconvolute(m = bulkRNAseq_ABIS, sigMatrix = sigList)

## -----------------------------------------------------------------------------
# print cell type proportions for svr model on ABIS_S0 reference profile
decon$proportions$svr_ABIS_S0[1:5, 1:5]

## ----fig.retina = 1-----------------------------------------------------------
# plot cell type proportions for svr model on ABIS_S0 reference profile
plot_proportions(deconvoluted = decon, method = 'svr', signature = 'ABIS_S0')

## ----fig.retina = 1-----------------------------------------------------------
# plot cell type proportions
plot_deconvolute(deconvoluted = decon, scale = TRUE, labels = FALSE)

## -----------------------------------------------------------------------------
# benchmark methods by correlating estimated to measured cell type proportions
bench <- benchmark(deconvoluted = decon, ground_truth = groundTruth_ABIS)

## -----------------------------------------------------------------------------
# print metrics
head(bench$summary)

## -----------------------------------------------------------------------------
# print metrics
head(bench$rank)

## ----fig.retina = 1-----------------------------------------------------------
# plot regression for svr model on ABIS_S0 reference profile
plot_regress(benchmarked = bench, method = 'svr', signature = 'ABIS_S0')

## ----fig.retina = 1-----------------------------------------------------------
# plot pearson correlation between predictions and true proportions
plot_benchmark(benchmarked = bench, metric = 'pcc')

## ----results='hide'-----------------------------------------------------------
# deconvolute input data using selected methods and reference profile matrix
methods <- c('ols','nnls','qprog','rls','svr')
decon <- deconvolute(bulkRNAseq_ABIS, list(ABIS_S2 = sigMatrix_ABIS_S2), methods)

## -----------------------------------------------------------------------------
# correlation analysis
correl <- correlate(deconvoluted = decon)

## ----fig.retina = 1-----------------------------------------------------------
# correlation heatmap
plot_correlate(correlated = correl, method="heatmap", legend=TRUE)

## -----------------------------------------------------------------------------
# correlation mean summary statistics
head(correl$summary)

## -----------------------------------------------------------------------------
# deconvolution method ranking
head(correl$rank)

## -----------------------------------------------------------------------------
# print session info
sessionInfo()

